
/*
   *  Object %name    : CRYS_LLF_KMNG.h
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief  Key Management h file (API and structures )
   *
   *  \version 
   *  \author yevgenys
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "DX_VOS_Sem.h"
#include "KMNG_Defs.h"
#include "CRYS_RSA_Types.h"
#include "CRYS_DH.h"
#include "CRYS_DH_KG.h"
//#include "LLF_KMNG_ERR.h"
//#include "LLF_KMNG_DEF.h"
#include "LLF_KMNG.h"
#include "CRYS_HASH.h"
#include "CRYS_AES.h"
#include "CRYS_RND.h"
#include "CRYS_RSA_KG.h"
#include "CRYS_RSA_BUILD.h"
#include "CRYS_KMNG.h"
#include "dx_hw_defs.h"
#include "kmng_host_op_code.h"


/*------------------------------------------------
    FUNCTIONS
--------------------------------------------------*/


/**
 * @brief     This function creates key accroding to the type, usage and restriction,
 *            and then wraps it and its data with the RKEK or the key passed as a parameter. The wrapped data is 
 *            loaded into the the provided memory. If the keyRingKeyData_ptr is NULL , then the created key is wrapped with RKEK
 *            The password and password lenght for the new key are passwed in the newKeyData_ptr.
 *			  On exit, newKeyData_ptr holds the created key data and key Id
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key ring key data
 * @param[in/out] newKeyData_ptr - lpointer to the created key data
 * @param[in] keyInfo_ptr - information of the created key
 * @param[in] KeyGenData_ptr - buffer for the internal use of the function (if buffer not need, NULL must be passed as argument)
 *
 * @return     DxError:  
 *                        
 */

DxError_t LLF_KMNG_CreateKey(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
                             KMNG_KeyDataParameter_t *newKeyData_ptr,
                             KMNG_KeyConfiguration_t *keyConfiguration_ptr,
							 KMNG_BufferData_t		 *KeyGenData_ptr )
{
  /* error */
  DxError_t       error;
  
  if(keyRingKeyData_ptr == DX_NULL)
  {
    /* create the protection key of the Key Ring */
	error =  DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_RING_OP_CODE, 4,
	   							   /* PARAMETERS: */
								   DX_NULL,
								   newKeyData_ptr,
								   keyConfiguration_ptr,
								   DX_NULL );
  }
  else
  {
    /* create the user key  */
	error =  DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_OP_CODE, 4,
	   							   /* PARAMETERS: */
								   keyRingKeyData_ptr,
								   newKeyData_ptr,
								   keyConfiguration_ptr,
								   DX_NULL );
  }
  
  return error;
}
                          
                                


/**
 * @brief     This function imports a symmetric key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] importSymKey_ptr - symmtric key data to be imported
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportSymUserKey(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
									KMNG_KeyConfiguration_t *keyConfiguration_ptr,
									KMNG_KeyDataParameter_t *importedKeyData_ptr,
									DxUint8_t               *importSymKey_ptr)
{

    
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_SYM_KEY_OP_CODE, 4,
	   							 /* PARAMETERS: */
								 keyRingKeyData_ptr,
                                 keyConfiguration_ptr,
                                 importedKeyData_ptr,
                                 importSymKey_ptr );

}
                                       
                                    
                                    
/**
 * @brief     This function imports a RSA key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulo_ptr - modulo pointer
 * @param[in] moduloSizeInBytes - modulo size
 * @param[in] E_ptr - E pointer
 * @param[in] E_SizeInBytes - E size
 * @param[in] D_ptr - D pointer
 * @param[in] D_SizeInBytes - D size
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportRSAUserKey(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
									KMNG_KeyConfiguration_t *keyConfiguration_ptr,
									KMNG_KeyDataParameter_t *importedKeyData_ptr,
									DxUint8_t               *modulo_ptr,
									DxUint32_t               moduloSizeInBytes,
									DxUint8_t               *E_ptr,
									DxUint32_t               E_SizeInBytes,
									DxUint8_t               *D_ptr,
									DxUint32_t               D_SizeInBytes)
{

	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_RSA_KEY_OP_CODE, 9,
	   							 /* PARAMETERS: */
								    keyRingKeyData_ptr,
									keyConfiguration_ptr,
									importedKeyData_ptr,
                                    modulo_ptr,
									moduloSizeInBytes,
									E_ptr,
									E_SizeInBytes,
									D_ptr,
									D_SizeInBytes );

}
                                       

/**
 * @brief     This function imports a DH key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulus_ptr - pointer to the modulus number
 * @param[in] modulusSize - modulus size in bytes
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize - generator size in bytes
 * @param[in] pub_ptr - pub key pointer
 * @param[in] pub_Size - pub key size in bytes
 * @param[in] priv_ptr - priv key pointer
 * @param[in] priv_Size - priv size in bytes
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_ImportDHUserKey(KMNG_KeyDataParameter_t*		keyRingKeyData_ptr,
								   KMNG_KeyConfiguration_t*     keyConfiguration_ptr,
								   KMNG_KeyDataParameter_t*		importedKeyData_ptr,
								   DxUint8_t*					modulus_ptr,
								   DxUint32_t					modulusSize,
								   DxUint8_t*					generator_ptr,
								   DxUint32_t					generatorSize,
								   DxUint8_t*					pub_ptr,
								   DxUint32_t					pub_Size,
								   DxUint8_t*					priv_ptr,
								   DxUint32_t					priv_Size)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_DH_KEY_OP_CODE, 11,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
								   keyConfiguration_ptr,
								   importedKeyData_ptr,
								   modulus_ptr,
								   modulusSize,
								   generator_ptr,
								   generatorSize,
								   pub_ptr,
								   pub_Size,
								   priv_ptr,
								   priv_Size );
}/* end function */

/**
* @brief     This function exports a symmetric key from key ring
*
*
* @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
* @param[out] keyConfiguration_ptr - configuration of the exported key
* @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
* @param[out] keyData_ptr - symmetric key itself after decryption
* @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-exportable
* 
* @return     DxError:  
*                        
*/                                       
DxError_t _LLF_KMNG_ExportSymUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
									 KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
									 KMNG_KeyDataParameter_t*   exportKeyData_ptr,
									 DxUint8_t*                 keyData_ptr,
									 DxUint32_t*                keyDataLen_ptr,
									 DxUint32_t                 exportInternalFlag)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_SYM_KEY_OP_CODE, 6,
		/* PARAMETERS: */
		keyRingKeyData_ptr,
		keyConfiguration_ptr,
		exportKeyData_ptr,
		keyData_ptr,
		keyDataLen_ptr,
		exportInternalFlag );
}

/**
 * @brief     This function exports an RSA key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] modulo_ptr - modulo pointer
 * @param[out] moduloSize_ptr - modulo size
 * @param[out] E_ptr - N pointer
 * @param[out] E_Size_ptr - N size
 * @param[out] D_ptr - D pointer
 * @param[out] D_Size_ptr - D size
 * @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-exportable
 * 
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportRSAUserKey( KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
									 KMNG_KeyConfiguration_t *keyConfiguration_ptr,
									 KMNG_KeyDataParameter_t *exportKeyData_ptr,
									 DxUint8_t               *modulo_ptr,
									 DxUint32_t              *moduloSize_ptr,
									 DxUint8_t               *E_ptr,
									 DxUint32_t              *E_Size_ptr,
									 DxUint8_t               *D_ptr,
									 DxUint32_t              *D_Size_ptr,
                                     DxUint32_t              exportInternalFlag)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_RSA_KEY_OP_CODE, 10,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyConfiguration_ptr,
                                   exportKeyData_ptr,
								   modulo_ptr,
                                   moduloSize_ptr,
                                   E_ptr,
                                   E_Size_ptr,
                                   D_ptr,
                                   D_Size_ptr,
                                   exportInternalFlag );
}

/**
 * @brief     This function exports an DH key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[in] modulus_ptr - modulus data pointer
 * @param[in] modulusSize_ptr - modulus size in bytes
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize_ptr - generator size in bytes
 * @param[in] pub_ptr - pub pointer
 * @param[in] pub_Size_ptr - pub size in bytes
 * @param[in] priv_ptr - private pointer
 * @param[in] priv_Size_ptr - private size in bytes
 * @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-expoertable
 * 
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportDHUserKey(KMNG_KeyDataParameter_t*	keyRingKeyData_ptr,
									KMNG_KeyConfiguration_t*	keyConfiguration_ptr,
									KMNG_KeyDataParameter_t*	exportKeyData_ptr,
									DxUint8_t*					modulus_ptr,
									DxUint32_t*					modulusSize_ptr,
									DxUint8_t*					generator_ptr,
									DxUint32_t*					generatorSize_ptr,
									DxUint8_t*					pub_ptr,
									DxUint32_t*					pub_Size_ptr,
									DxUint8_t*					priv_ptr,
									DxUint32_t*					priv_Size_ptr,
									DxUint32_t					exportInternalFlag)
{
    
    /* opcode */
    DxUint32_t  opcode;

    if(exportInternalFlag == 0)
    {
      opcode = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_DH_KEY_OP_CODE;
    }
    else
    {
      opcode = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_READ_DH_KEY_OP_CODE;
    }
	return DX_HW_Dispatcher_ptr( opcode, 12,
	   							 /* PARAMETERS: */
                                    keyRingKeyData_ptr,
                                    keyConfiguration_ptr,
                                    exportKeyData_ptr,
									modulus_ptr,
									modulusSize_ptr,
									generator_ptr,
									generatorSize_ptr,
									pub_ptr,
									pub_Size_ptr,
									priv_ptr,
									priv_Size_ptr,
									exportInternalFlag );
}

/**
 * @brief     checks the authorization of the given key, by checking the hash of the password
 *            stored in the key data and the given password
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] authorizeKeyData_ptr - pointer to the data of the key to be authorized
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_AuthorizeKey(KMNG_KeyDataParameter_t*  keyRingKeyData_ptr,
                                KMNG_KeyDataParameter_t*  authorizeKeyData_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_AUTHORIZE_KEY_OP_CODE, 2,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   authorizeKeyData_ptr );
}
                                                                          
                                       
/**
 * @brief     This function transfers key from one key ring into another
 * (actually decrypts it and then enctypts it with destination key ring key).
 *
 * @param[in] srcKeyRingKeyData_ptr - pointer to the key data of the source Key Ring
 * @param[in] srcKeyData_ptr - key data of the source key to be transfered
 * @param[in] destRingKeyData_ptr - pointer to the key data of the destination Key Ring
 * @param[out] destKeyData_ptr - data of key transferred to the destination key ring
 *
 * @return     DxError:  
 *                        
 */                                
DxError_t LLF_KMNG_TransferKey(KMNG_KeyDataParameter_t*  srcKeyRingKeyData_ptr,
							   KMNG_KeyDataParameter_t*  srcKeyData_ptr,
							   KMNG_KeyDataParameter_t*  destRingKeyData_ptr,
							   KMNG_KeyDataParameter_t*  destKeyData_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_TRANSFER_KEY_OP_CODE, 4,
	   							 /* PARAMETERS: */
                                   srcKeyRingKeyData_ptr,
                                   srcKeyData_ptr,
                                   destRingKeyData_ptr,
                                   destKeyData_ptr );
}
                                
                                
/**
 * @brief     This function prepares the AES contest for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]aesContext_ptr - memory to stores created AES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateAESKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_AES_WrappedKey_t		AES_WrappedKey_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_AES_KEY_OP_CODE, 3,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyData_ptr,
                                   AES_WrappedKey_ptr );
}

/**
 * @brief     This function prepares the DES wrap key for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]DES_WrappedKey_ptr - memory to stores created DES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDESKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_DES_WrappedKey_t		DES_WrappedKey_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DES_KEY_OP_CODE, 3,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyData_ptr,
                                   DES_WrappedKey_ptr );

}
/**
 * @brief     This function prepares the HMAC contest for CRYS function usage based on the key ring HMAC key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]HMAC_WrappedKey_ptr - memory to stores created HMAC context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateHMACKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_HMAC_WrappedKey_t	HMAC_WrappedKey_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_HMAC_KEY_OP_CODE, 3,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyData_ptr,
                                   HMAC_WrappedKey_ptr );
}
                                     
/**
 * @brief     This function prepares the RSA contest for CRYS function usage based on the key ring RSA key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]RSA_WrappedKey_ptr - memory to stores created RSA context  
 * @param[out]RSA_WrappedKeyLen_ptr - memory to stores created RSA context  
 *	The function gets output bufer and output buffer length because the out put
 *   can be private key or public key which have different size
 *                        
 * @return     DxError:  
 *                        
 *	MAJOR ASSUMPTION - RSA_WrappedKey_ptr can be the same buffer as the keyData_ptr buffer pointer                        
 *                        
 */
DxError_t LLF_KMNG_ActivateRSAKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t*   keyData_ptr,
								  KMNG_RsaKeyOperationType_t rsaOperationType,
								  DxUint8_t*				RSA_WrappedKey_ptr,
								  DxUint32_t*				RSA_WrappedKeyLen_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_RSA_KEY_OP_CODE, 5,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyData_ptr,
                                   rsaOperationType, 
                                   RSA_WrappedKey_ptr, 
                                   RSA_WrappedKeyLen_ptr );

}
                                    
                                    
/**
 * @brief     This function prepares the DH contest for CRYS function usage based on the key ring DH key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]dhContext_ptr - memory to stores created DH context  
 * @param[out]DH_WrappedKeyLen_ptr - the length of the wraped key 
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDHKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
								 KMNG_KeyDataParameter_t*   keyData_ptr,
								 KMNG_DH_WrappedKey_t		DH_WrappedKey_ptr,
								 DxUint32_t*				DH_WrappedKeyLen_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DH_KEY_OP_CODE, 4,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   keyData_ptr,
                                   DH_WrappedKey_ptr, 
                                   DH_WrappedKeyLen_ptr );
}

/**
 * @brief     This function retrive the configuration of key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_RetrieveKeyConfiguration(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
											KMNG_KeyDataParameter_t *retrivedKeyData_ptr,
											KMNG_KeyConfiguration_t *keyConfiguration_ptr)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_RETRIEVE_KEY_CONFIGURATION_OP_CODE, 3,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   retrivedKeyData_ptr,
                                   keyConfiguration_ptr );
}/* end function */


/**
 * @brief     This function updates the user specific data in the key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] keyData_ptr - symmetric key itself after decryption
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_UpdateKeyUserData(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
								     KMNG_KeyDataParameter_t *updateKeyData_ptr,
								     KMNG_UserSpecificKeyData_t   UserSpecificKeyData)
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_UPDATE_KEY_SPECIF_DATA_OP_CODE, 3,
	   							 /* PARAMETERS: */
                                   keyRingKeyData_ptr,
                                   updateKeyData_ptr,
                                   UserSpecificKeyData );
}/* end function */

